<?php
/**
 * ===================================================================================
 * Grade Submission Page (Mobile Responsive)
 * ===================================================================================
 *
 * File: grade_submission.php
 * Location: /homework_portal/teacher/grade_submission.php
 *
 * Purpose:
 * 1. Allows a teacher to grade a submission via a responsive interface.
 * 2. Safely handles date formatting.
 * 3. Adapts for mobile, tablet, and desktop screens.
 */

require_once '../config.php'; 

// --- Security Check & Initial Setup ---
if (!isset($_SESSION['teacher_id'])) {
    header('Location: login.php');
    exit();
}

$teacher_id = $_SESSION['teacher_id'];
$teacher_full_name = e($_SESSION['teacher_full_name'] ?? 'Teacher');
$submission_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$errors = [];
$success_message = '';

if ($submission_id <= 0) {
    header('Location: view_submissions.php');
    exit();
}

// --- Handle Form Submission to Grade/Update Grade ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['grade_submission'])) {
    $grade = trim($_POST['grade'] ?? '');
    $feedback = trim($_POST['feedback'] ?? '');
    
    // Fetch the existing graded file path first
    $stmt = $pdo->prepare("SELECT graded_file_path FROM submissions WHERE id = ?");
    $stmt->execute([$submission_id]);
    $current_graded_file = $stmt->fetchColumn();
    $graded_file_path = $current_graded_file;

    if (isset($_FILES['graded_file']) && $_FILES['graded_file']['error'] == UPLOAD_ERR_OK) {
        $upload_dir_physical = __DIR__ . '/../uploads/graded/';
        if (!is_dir($upload_dir_physical)) { mkdir($upload_dir_physical, 0755, true); }

        // Delete old file if it exists
        if (!empty($current_graded_file) && file_exists($upload_dir_physical . basename($current_graded_file))) {
            unlink($upload_dir_physical . basename($current_graded_file));
        }
        
        $file_name = "graded_{$submission_id}_" . uniqid() . '_' . basename($_FILES['graded_file']['name']);
        $target_file = $upload_dir_physical . $file_name;

        if (move_uploaded_file($_FILES['graded_file']['tmp_name'], $target_file)) {
            $graded_file_path = 'uploads/graded/' . $file_name;
        } else {
            $errors[] = "Failed to upload new graded file.";
        }
    }

    if (empty($grade)) $errors[] = "A grade is required.";

    if (empty($errors)) {
        try {
            $update_sql = "UPDATE submissions SET grade = ?, teacher_feedback = ?, graded_file_path = ?, graded_at = NOW(), grade_seen_by_student = 0 WHERE id = ?";
            $pdo->prepare($update_sql)->execute([$grade, $feedback, $graded_file_path, $submission_id]);
            $success_message = "Submission has been graded successfully!";
        } catch (PDOException $e) {
            error_log($e->getMessage());
            $errors[] = "Database error: Could not update grade.";
        }
    }
}

// --- Fetch all data for display ---
try {
    // Fetch submission details
    $sql = "
        SELECT 
            sub.id, sub.file_path, sub.submitted_at, sub.grade, sub.teacher_feedback, sub.graded_file_path, sub.graded_at,
            a.title AS assignment_title,
            s.full_name AS student_name
        FROM submissions sub
        JOIN assignments a ON sub.assignment_id = a.id
        JOIN students s ON sub.student_id = s.id
        WHERE sub.id = ? AND a.teacher_id = ?
    ";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$submission_id, $teacher_id]);
    $submission = $stmt->fetch();

    if (!$submission) {
        header('Location: view_submissions.php?error=Submission not found or permission denied.');
        exit();
    }

    // Fetch counts for sidebar badges
    $unread_messages_count = $pdo->query("SELECT COUNT(id) FROM teacher_messages WHERE teacher_id = {$teacher_id} AND is_read = 0 AND sender_type = 'admin'")->fetchColumn();
    $new_submission_count = $pdo->query("SELECT COUNT(sub.id) FROM submissions sub JOIN assignments a ON sub.assignment_id = a.id LEFT JOIN teacher_seen_submissions tss ON sub.id = tss.submission_id AND tss.teacher_id = {$teacher_id} WHERE a.teacher_id = {$teacher_id} AND tss.submission_id IS NULL")->fetchColumn();

} catch (PDOException $e) {
    error_log($e->getMessage());
    die("Database error: Could not fetch submission details.");
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Grade Submission - Teacher Portal</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&family=Baloo+2:wght@700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Poppins', sans-serif; background-color: #f0f9ff; }
        .font-brand { font-family: 'Baloo 2', cursive; }
        .sidebar-link { display: flex; align-items: center; padding: 0.75rem 1rem; border-radius: 0.5rem; transition: background-color 0.2s, color 0.2s; font-weight: 500;}
        .sidebar-link:hover, .sidebar-link.active { background-color: #e0f2fe; color: #0284c7; font-weight: 600; }
        .notification-badge { display: inline-flex; align-items: center; justify-content: center; min-width: 22px; height: 22px; padding: 0 4px; border-radius: 9999px; background-color: #ef4444; color: white; font-size: 0.75rem; font-weight: bold; }
    </style>
</head>
<body class="flex flex-col min-h-screen">
    <div class="relative min-h-screen md:flex">
        <!-- Mobile menu button -->
        <div class="md:hidden flex justify-between items-center bg-white p-4 shadow-md">
             <a href="dashboard.php">
                <h1 class="text-3xl font-bold text-red-600 font-sans">TTTEDDDEL</h1>
            </a>
            <button id="mobile-menu-button" class="text-gray-500 hover:text-blue-600 focus:outline-none">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16m-7 6h7"></path></svg>
            </button>
        </div>

        <!-- Sidebar Navigation -->
        <aside id="sidebar" class="w-64 bg-white shadow-md absolute inset-y-0 left-0 transform -translate-x-full md:relative md:translate-x-0 transition-transform duration-300 ease-in-out z-30">
            <div class="p-4 border-b text-center">
                <a href="dashboard.php">
                    <h1 class="text-4xl font-bold text-red-600 tracking-widest font-sans">TTTEDDDEL</h1>
                    <p class="text-xs font-semibold text-blue-800 mt-1">Teacher-Student Interactive System</p>
                    <p class="text-xs text-blue-800">by Ele Infocomm Technologies</p>
                </a>
            </div>
            <nav class="p-4 space-y-2">
                <a href="dashboard.php" class="sidebar-link"><span>Dashboard</span></a>
                <a href="inbox.php" class="sidebar-link flex justify-between items-center">
                    <span>Inbox</span>
                    <?php if ($unread_messages_count > 0): ?><span class="notification-badge"><?php echo $unread_messages_count; ?></span><?php endif; ?>
                </a>
                <a href="manage_students.php" class="sidebar-link"><span>My Students</span></a>
                <a href="post_material.php" class="sidebar-link"><span>Post Material</span></a>
                <a href="my_materials.php" class="sidebar-link"><span>My Posted Materials</span></a>
                <a href="view_submissions.php" class="sidebar-link active"><span>View Submissions</span></a>
            </nav>
        </aside>

        <!-- Main Content Area -->
        <div class="flex-1 flex flex-col overflow-hidden">
            <header class="bg-white shadow-sm border-b hidden md:flex"><div class="flex items-center justify-end p-4 w-full"><span class="text-gray-600 mr-4">Welcome, <?php echo $teacher_full_name; ?>!</span><a href="../logout.php" class="ml-2 py-2 px-4 bg-teal-600 text-white font-semibold rounded-lg shadow-md hover:bg-teal-700">Logout</a></div></header>
            
            <main class="flex-1 overflow-x-hidden overflow-y-auto bg-gray-100 p-4 md:p-6">
                <div class="mb-6">
                    <a href="view_submissions.php" class="inline-flex items-center text-teal-600 hover:text-teal-800 font-medium">
                        <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path></svg>
                        Back to All Submissions
                    </a>
                </div>

                <div class="bg-white p-6 rounded-lg shadow-lg">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                        <div class="border-r-0 md:border-r md:pr-8">
                            <h1 class="text-2xl font-bold text-gray-800">Grade Submission</h1>
                            <div class="mt-6 space-y-4 text-sm">
                                <div><h3 class="font-semibold text-gray-600">Student:</h3><p class="text-gray-800 text-base"><?php echo e($submission['student_name']); ?></p></div>
                                <div><h3 class="font-semibold text-gray-600">Assignment:</h3><p class="text-gray-800 text-base"><?php echo e($submission['assignment_title']); ?></p></div>
                                <div>
                                    <h3 class="font-semibold text-gray-600">Submitted At:</h3>
                                    <p class="text-gray-800 text-base">
                                        <?php echo !empty($submission['submitted_at']) ? date('F j, Y, g:i a', strtotime($submission['submitted_at'])) : "Date not available"; ?>
                                    </p>
                                </div>
                                <div><h3 class="font-semibold text-gray-600">Student's Submitted File:</h3><a href="../<?php echo e($submission['file_path']); ?>" download class="text-blue-600 hover:underline">Download Submission</a></div>
                            </div>
                        </div>
                        <div>
                            <?php if ($success_message): ?><div class="p-4 mb-4 text-sm text-green-700 bg-green-100 rounded-lg"><?php echo e($success_message); ?></div><?php endif; ?>
                            <?php if (!empty($errors)): ?><div class="p-4 mb-4 text-sm text-red-700 bg-red-100 rounded-lg"><ul class="list-disc list-inside"><?php foreach ($errors as $e) echo "<li>" . e($e) . "</li>"; ?></ul></div><?php endif; ?>
                            
                            <form action="grade_submission.php?id=<?php echo $submission_id; ?>" method="POST" enctype="multipart/form-data" class="space-y-6">
                                <div>
                                    <label for="grade" class="block text-sm font-medium text-gray-700">Grade / Score</label>
                                    <input type="text" name="grade" id="grade" value="<?php echo e($submission['grade'] ?? ''); ?>" class="mt-1 block w-full p-2 border border-gray-300 rounded-md" required placeholder="e.g., A+, 85%, 25/30">
                                </div>
                                <div>
                                    <label for="feedback" class="block text-sm font-medium text-gray-700">Feedback for Student</label>
                                    <textarea name="feedback" id="feedback" rows="5" class="mt-1 block w-full p-2 border border-gray-300 rounded-md" placeholder="Provide constructive feedback..."><?php echo e($submission['teacher_feedback'] ?? ''); ?></textarea>
                                </div>
                                <div>
                                    <label for="graded_file" class="block text-sm font-medium text-gray-700">Upload Marked/Graded File (Optional)</label>
                                    <input type="file" name="graded_file" id="graded_file" class="mt-1 block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:font-semibold file:bg-sky-50 file:text-sky-700 hover:file:bg-sky-100">
                                    <?php if ($submission['graded_file_path']): ?><p class="text-sm text-gray-600 mt-2">Current graded file: <a href="../<?php echo e($submission['graded_file_path']); ?>" download class="text-blue-600 hover:underline">Download</a></p><?php endif; ?>
                                </div>
                                <div class="flex justify-end pt-4 border-t">
                                    <button type="submit" name="grade_submission" class="w-full sm:w-auto px-6 py-2 bg-teal-600 text-white font-semibold rounded-lg shadow hover:bg-teal-700">Save Grade</button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <footer class="text-center py-6 bg-gray-100 border-t mt-auto">
        <p class="text-xs text-gray-500">Built By Ele InfoComm Technologies- Kenya | <a href="http://web.e-ict.co.ke" target="_blank" class="hover:underline">web.e-ict.co.ke</a> | <a href="mailto:info@e-ict.co.ke" class="hover:underline">info@e-ict.co.ke</a> | +254202000121 | +254 722 215810 | 2025 All right reserved.</p>
    </footer>

    <script>
        const btn = document.getElementById('mobile-menu-button');
        const sidebar = document.getElementById('sidebar');

        btn.addEventListener('click', () => {
            sidebar.classList.toggle('-translate-x-full');
        });
    </script>
</body>
</html>
